//
//  CMS_HTTP.swift
//  Composr Mobile SDK
//
//  Created by Aaswini on 06/08/14.
//  Copyright (c) 2014 Aaswini. All rights reserved.
//
/*
 
 CMS_HTTP
 
 string rawurlencode(string str) - see the PHP manual
 object json_decode(string value) - see the PHP manual
 string json_encode(object value) - see the PHP manual
 string get_base_url - gets base URL to ocPortal site, defined for the project in a string resource of constant
 string build_url(map params, string zone) - creates <base_url>/<zone>/index.php?<params...>
 string http_download_file(string url, bool triggerError, map postParams, int timeoutInSeconds) - if triggerError is true, a failure will show a generic error alert. If postParams is not null/nil then it will be an HTTP post.
 bool has_network_connection() - base the code on what we use for Fishin' Mobile, but make sure it is generic
 
 */

import UIKit
import Foundation

typealias CMSCompletedDownloadBlock = (response: String) -> Void

class CMS_HTTP: NSObject {
    
    /**
     *  URL-encode according to RFC 3986
     *
     *  @param str The URL to be encoded.
     *
     *  @return Returns a string in which all non-alphanumeric characters except -_.~ have been replaced with a percent (%) sign followed by two hex digits. This is the encoding described in » RFC 3986 for protecting literal characters from being interpreted as special URL delimiters, and for protecting URLs from being mangled by transmission media with character conversions (like some email systems).
     */
    class func rawurlencode(str: String) -> String {
        return str.urlEncodedString()
    }
    
    /**
     *  Takes a JSON encoded string and converts it into a PHP variable
     *
     *  @param value The json string being decoded
     *
     *  @return Returns the value encoded in json
     */
    class func json_decode(value: String) -> AnyObject {
        return value.objectFromJSONString()
    }
    
    /**
     *  Returns a string containing the JSON representation of value.
     *
     *  @param value The value being encoded.
     *
     *  @return Returns a JSON encoded string on success
     */
    class func json_encode(value: AnyObject) -> String {
        return value.JSONString()
    }
    
    /**
     *  Returns the base url used by web services in the app
     *
     *  @return Returns the base url used by web services in the app as String
     */
    class func get_base_url() -> String {
        return k_WEBSERVICES_DOMAIN
    }
    
    /**
     *  creates <base_url>/<zone>/index.php?<params...>
     *
     *  @param params parameters to be appended to url
     *  @param zone   zone to be used
     *
     *  @return built url from zone and parameter values
     */
    class func build_url(params: [NSObject : AnyObject], zone: String) -> String {
        var url = ""
        url += k_WEBSERVICES_DOMAIN
        url += "/\(zone)/index.php?"
        for key: String in params.allKeys() {
            url += "&\(key)=\(params[key] as! String)"
        }
        return url
    }
    
    /**
     *  Downloads string contents from a url
     *
     *  @param url              url
     *  @param triggerError     to show error or not
     *  @param postParams       post parameters
     *  @param timeoutInSeconds timeout for the request
     *
     *  @return returns the response for the url request
     */
    class func http_download_file(url: String, triggerError: Bool, postParams: [NSObject : AnyObject], timeoutInSeconds: Int, completionHandler: CMSCompletedDownloadBlock) {
        var manager = AFHTTPRequestOperationManager.manager()
        manager.POST(url, parameters: postParams, success: {(operation: AFHTTPRequestOperation, responseObject: AnyObject) -> Void in
            print("JSON: \(responseObject)")
            completionHandler(operation.responseString)
        }, failure: {(operation: AFHTTPRequestOperation, error: NSError) -> Void in
            if error && triggerError {
                UIAlertView.showWithError(error)
            }
        })
    }
    
    /**
     *  Returns status of internet connection
     *
     *  @return YES if connected. NO otherwise.
     */
    class func has_network_connection() -> Bool {
        if NSProcessInfo.processInfo().arguments().indexOf("work_offline") != NSNotFound {
            return false
        }
        var connectionMonitor = Reachability.reachabilityForInternetConnection()
        return (connectionMonitor.currentReachabilityStatus() != NotReachable)
    }

}